#! /usr/bin/perl -w

use strict;
use warnings FATAL => qw( all );

use Time::Local;

sub LocalDateString($);
sub DurationStringSec($);
sub DurationStringMin($);

my %mp_mode_name = 
(
	0 => 'Standard',
	1 => 'Storage',
	3 => 'Range',
	4 => 'Performance'
);

my @afield_S =
(
	'SOC',
	'mikm',
	'volts',
	'amps',
	'charge_state',
	'charge_mode',
	'iRange',
	'eRange',
	'charge_limit',
	'charge_duration',
	'charge_b4',
	'kwh_charge',
	'charge_substate_n',
	'charge_state_n',
	'charge_mode_n',
	'timer_mode',
	'timer_start',
	'timer_1stale',
	'CAC',
	'min_ctp_to_full',
	'min_ctp_to_limit',
	'acc_range_limit',
	'acc_soc_limit',
	'cooling',
	'cooldown_tbattery',
	'cooldown_timelimit',
	'acc_charge_estimate'
);

my @afield_D =
(
	'doors1',
	'doors2',
	'lockstate',
	'temp_pem',
	'temp_motor',
	'temp_battery',
	'trip',
	'odometer',
	'speed',
	'park',
	'temp_ambient',
	'doors3',
	'state_temps',
	'state_ambient',
	'acc_battery',
	'doors4',
	'acc_battery_ref',
	'doors5'
);

my @afield_L =
(
	'latitude',
	'longitude',
	'direction',
	'elevation', # called altitude in the OVMS code
	'gps-lock',
	'gps-stale'
);

my @afield_W =
(
	'f1p',
	'f1t',
	'r1p',
	'r1t',
	'f2p',
	'f2t',
	'r2p',
	'r2t',
	'stale'
);

my @afield_H =
(
	'crash_type',
	'const_0',
	'const_2592000',
	'OVMS_version',
	'crash_count',
	'crash_reason',
	'checkpoint'
);

my @afield_F = # firmware version, signal strength, etc
(
	'OVMS-version',
	'VIN',
	'net-signal-quality',
	'can-write',
	'car-type',
	'gsm-provider',
);

my %mp_command_name =
(
	'1' => 'Get Feature',
	'2' => 'Set Feature',
	'3' => 'Get Parameter',
	'4' => 'Set Parameter',
);

my @afield_C = # command
(
	'command',
);

my @afield_c = # command response
(
	'command',
);

my @afield_h =
(
	'index',
	'secs_elapsed',
	'type',
	'const_zero',
	'const_31536000',
	'start_time',
	'duration',
);

my @afield_h_drive =
(
	'index',
	'secs_elapsed',
	'type',
	'const_zero',
	'const_31536000',
	'start_time',
	'duration',
	'mode',
	'start_latitude',
	'start_longitude',
	'end_latitude',
	'end_longitude',
	'distance',
	'start_SOC',
	'start_idealrange',
	'end_SOC',
	'end_idealrange'
);

my @afield_h_charge =
(
	'index',
	'secs_elapsed',
	'type',
	'const_zero',
	'const_31536000',
	'start_time',
	'duration',
	'mode',
	'charge_latitude',
	'charge_longitude',
	'charge_voltage',
	'charge_current',
	'charge_result',
	'start_SOC',
	'start_idealrange',
	'end_SOC',
	'end_idealrange',
	'cac'
);

my $unit_dist = 'mi/km?';
my $unit_speed = 'mph/kph?';
my $abbrev_dist = 'M/K?';
while (<>)
{
	chomp;
	
    # 2014-09-15 13:59:10	58101	#146 C rx msg S 96,M,-1,0,done,standard,177,156,32,103,100,12,7,4,0,0,0,-1,149.69,-3,-1,0,0,-1,27,57,117
	next unless /^(\d+)-(\d+)-(\d+) (\d+):(\d+):(\d+)\t(\d+)\t#(\d+) (\w) rx msg (\w)\s+(.+)$/;
	
	my $timestamp = timegm($6,$5,$4,$3,$2-1,$1-1900);
	my $date = LocalDateString($timestamp);
	my $type = $10;
	my @data = split(',', $11);

	my $entry = ();
	if ($type eq "S")
	{
		foreach my $field (@afield_S)
		{
			$entry->{$field} = shift @data;
		}
		if ($entry->{'mikm'} ne '')
		{
			$abbrev_dist = $entry->{'mikm'};
			if ($abbrev_dist eq 'M')
			{
				$unit_dist = 'mi';
				$unit_speed = 'mph';
			}
			elsif ($abbrev_dist eq 'K')
			{
				$unit_dist = 'km';
				$unit_speed = 'kph';
			}
		}
		my $durationCharge = DurationStringMin($entry->{'charge_duration'});
		print "$date $type $entry->{charge_mode} $entry->{SOC}%, $entry->{iRange} I$entry->{mikm}, CAC $entry->{CAC} Ah, $durationCharge, $entry->{kwh_charge} kWh, $entry->{volts}V/$entry->{amps}A";
		if ($entry->{'acc_charge_estimate'} > 0)
		{
			my $durationCTP = DurationStringMin($entry->{'acc_charge_estimate'});
			print ", CTP $durationCTP";
		}
		print "\n";
	}
	elsif ($type eq 'D')
	{
		foreach my $field (@afield_D)
		{
			$entry->{$field} = shift @data;
		}
		my $odometer = sprintf("%.1f", $entry->{odometer} * 0.1);
		my $trip = sprintf("%.1f", $entry->{trip} * 0.1);
		print "$date $type odo $odometer, trip $trip, Amb $entry->{temp_ambient}C, Batt $entry->{temp_battery}C, PEM $entry->{temp_pem}C, Motor $entry->{temp_motor}C";
		if ($entry->{'park'} > 0)
		{
			my $durationParked = DurationStringSec($entry->{'park'});
			print ", parked $durationParked";
		}
		else
		{
			print ", $entry->{speed} $unit_speed";
		}
		print "\n";
	}
	elsif ($type eq 'h')
	{
		my $index = $data[0];
		my $typelog = $data[2];
		if ($typelog eq '*-Log-Drive')
		{
			foreach my $field (@afield_h_drive)
			{
				$entry->{$field} = shift @data;
			}
			my $dateStart = LocalDateString($entry->{'start_time'});
			my $dateEnd = LocalDateString($entry->{'start_time'} + $entry->{'duration'});
			my $duration = DurationStringSec($entry->{'duration'});
			my $mode = $entry->{'mode'};
			if ($mp_mode_name{$mode})
			{
				$mode = $mp_mode_name{$mode};
			}
			print "$date $type Drive Log $index, $dateStart $dateEnd ($duration), $mode, $entry->{start_SOC}% -> $entry->{end_SOC}%, $entry->{start_idealrange} -> $entry->{end_idealrange} I$abbrev_dist, $entry->{distance} $unit_dist\n";
		}
		elsif ($typelog eq '*-Log-Charge')
		{
			foreach my $field (@afield_h_charge)
			{
				$entry->{$field} = shift @data;
			}
			my $dateStart = LocalDateString($entry->{'start_time'});
			my $dateEnd = LocalDateString($entry->{'start_time'} + $entry->{'duration'});
			my $duration = DurationStringSec($entry->{'duration'});
			my $mode = $entry->{'mode'};
			if ($mp_mode_name{$mode})
			{
				$mode = $mp_mode_name{$mode};
			}
			print "$date $type Charge Log $index, $dateStart $dateEnd ($duration), $mode, $entry->{start_SOC}% -> $entry->{end_SOC}%, $entry->{start_idealrange} -> $entry->{end_idealrange} I$abbrev_dist, CAC $entry->{cac} Ah\n";
		}
		else
		{
			print "$date $type $index $typelog\n";
		}
	}
	elsif ($type eq 'L')
	{
		# GPS/Location record
		foreach my $field (@afield_L)
		{
			$entry->{$field} = shift @data;
		}
		my $latitude = $entry->{'latitude'};
		if ($latitude >= 0)
		{
			$latitude .= 'N';
		}
		else
		{
			$latitude = -$latitude;
			$latitude .= 'S';
		}
		my $longitude = $entry->{'longitude'};
		if ($longitude >= 0)
		{
			$longitude .= 'E';
		}
		else
		{
			$longitude = -$longitude;
			$longitude .= 'W';
		}
		print "$date $type $latitude, $longitude, elevation $entry->{elevation}m, lock $entry->{'gps-lock'}, stale $entry->{'gps-stale'}";
		print "\n";
	}
	elsif ($type eq 'F')
	{
		# Tire Pressure Monitoring System
		foreach my $field (@afield_F)
		{
			$entry->{$field} = shift @data;
		}
		print "$date $type " , join(', ',
			"OVMS v$entry->{'OVMS-version'}",
			"VIN $entry->{'VIN'}",
			"signal quality $entry->{'net-signal-quality'}",
			"CAN write $entry->{'can-write'}",
			"car type $entry->{'car-type'}",
			"GSM provider $entry->{'gsm-provider'}",
			);
		print "\n";
	}
	elsif ($type eq 'W')
	{
		# Tire Pressure Monitoring System
		foreach my $field (@afield_W)
		{
			$entry->{$field} = shift @data;
		}
		print "$date $type " , join(', ',
			"front1 $entry->{'f1p'}psi $entry->{'f1t'}C",
			"rear1 $entry->{'r1p'}psi $entry->{'r1t'}C",
			"front2 $entry->{'f2p'}psi $entry->{'f2t'}C",
			"rear2 $entry->{'r2p'}psi $entry->{'r2t'}C",
			"stale $entry->{'stale'}"
			);
		print "\n";
	}
	elsif ($type eq 'V')
	{
		# capabilities
		print "$date $type " , join(',', @data);
		print "\n";
	}
	elsif ($type eq 'H')
	{
		# Crash Report
		foreach my $field (@afield_H)
		{
			$entry->{$field} = shift @data;
		}
		print "$date $type " , join(', ',
			"$entry->{'crash_type'}",
			"v$entry->{'OVMS_version'}",
			"count $entry->{'crash_count'}",
			"reason $entry->{'crash_reason'}",
			"checkpoint $entry->{'checkpoint'}"
			);
		print "\n";
	}
	elsif ($type eq 'P')
	{
		# NYI
		print "$date $type $11\n";
	}
	elsif ($type eq 'C')
	{
		# command
#		print "$date $type $11\n";
		foreach my $field (@afield_C)
		{
			$entry->{$field} = shift @data;
		}
		my $kind = $mp_command_name{$entry->{'command'}};
		if ($entry->{'command'} == 1 || $entry->{'command'} == 3)
		{
			print "$date $type Send Command: $kind\n";
		}
		elsif ($entry->{'command'} == 2 || $entry->{'command'} == 4)
		{
			my $value = $data[1] ? $data[1] : '';
			print "$date $type Send Command: $kind $data[0] to '$value'\n";
		}
		else
		{
			print "$date $type unknown command ($entry->{'command'}) " ,
				join(',', @data),
				"\n";
		}
	}
	elsif ($type eq 'c')
	{
		# command response
#		print "$date $type $11\n";
		foreach my $field (@afield_c)
		{
			$entry->{$field} = shift @data;
		}
		my $kind = $mp_command_name{$entry->{'command'}};
		if ($entry->{'command'} == 1 || $entry->{'command'} == 3)
		{
			my $const_0 = shift @data;
			my $index = shift @data;
			my $iMost = (shift @data) - 1;
			my $value = shift @data;
			if (!defined($value))
			{
				$value = '';
			}
			print "$date $type $kind $index of $iMost = $value\n";
		}
		elsif ($entry->{'command'} == 2 || $entry->{'command'} == 4)
		{
			print "$date $type $kind result $data[0]";
			print ' (' , ($data[0] == 0 ? 'success' : 'failure') , ')';
			print "\n";
		}
		else
		{
			print "$date $type unknown command ($entry->{'command'}) result" ,
				join(',', @data),
				"\n";
		}
	}
	elsif ($type eq 'p')
	{
		# NYI
		print "$date $type $11\n";
	}
	elsif ($type eq 'g')
	{
		# NYI
		print "$date $type $11\n";
	}
	else
	{
		print "$date $type $11\n";
		die 'unknown record type: $type';
	}
}

sub LocalDateString($)
{
	my ($time) = @_;
	
	my ($sec, $min, $hour, $mday, $mon, $year) = localtime($time);
	return sprintf("%02d/%02d/%04d-%02d:%02d:%02d", $mon + 1, $mday, $year + 1900, $hour, $min, $sec);
}

sub DurationStringSec($)
{
	my ($duration) = @_;
	
	if ($duration < 0)
	{
		return $duration;
	}

	my $sec = $duration % 60;
	$duration -= $sec;
	$duration /= 60;

	my $min = $duration % 60;
	$duration -= $min;
	$duration /= 60;
	
	my $hour = $duration;
	
	return sprintf('%d:%02d:%02d', $hour, $min, $sec);
}

sub DurationStringMin($)
{
	my ($duration) = @_;
	
	if ($duration < 0)
	{
		return $duration;
	}

	my $min = $duration % 60;
	$duration -= $min;
	$duration /= 60;
	
	my $hour = $duration;
	
	return sprintf('%d:%02d', $hour, $min);
}
